/*
* Program send200.c
*

Uses interrupt service routine to note interrupt from printer port.
The interrupt is caused by a negative on /ACK input on Printer Port.

*/

/************************************************************************

gump:sonycd[11]% send200 -w -t 30000 play_dt 1 1
player 0: play
player 0: goto_disc disc 1
player 0: unloading
player 0: now_at_disc disc 1
player 0: displaying_disc disc 1
player 0: ready
player 0: playing disc 1 trk 1 len 02:51
gump:sonycd[12]% send200 -w -t 5000 play_dt 1 1
player 0: playing disc 1 trk 1 len 02:51
gump:sonycd[13]% send200 -w -t 5000 play_dt 1 1
player 0: playing disc 1 trk 1 len 02:51
gump:sonycd[14]% send200 pause
player 0: pause
gump:sonycd[15]% send200 mode
player 0: mode disc 1 track 1 discs_known one_disc
gump:sonycd[16]% send200 -w -t 5000 play_dt 1 1
player 0: play
player 0: playing disc 1 trk 1 len 02:51
gump:sonycd[17]% send200 mode
player 0: mode disc 1 track 1 playing discs_known one_disc
gump:sonycd[18]% send200 pause
player 0: pause
gump:sonycd[19]% 

gump:sonycd[30]% send200 play
.
.
.
gump:sonycd[17]% send200 -w -t ???? mode
player 0: mode disc 13 track 1 playing discs_known one_disc
player 0: 30s_to_eot
player 0: playing disc 13 trk 2 len 04:10

************************************************************************/

#include <stdio.h>
#include <bios.h>
#include <dos.h>
#include <string.h>
#include <sys\timeb.h>

#define DATA 0x0378
#define STATUS DATA+1
#define CONTROL DATA+2

#define TRUE 1
#define FALSE 0

char *rcode2text(char *code);
char *text2scode(int player, int argc, char **argv);

static int debug = 0;
static int calibrate = 0;
static int exit_on_first_response = 1;
static long timeout_msec = 10000;
static int player = 1;

int intlev=0x0f;  /* interrupt level associated with IRQ7 */
void interrupt far (*oldfunc)();
int int_occurred=FALSE;  /* Note global definitions */
long loops = 0L;

static long unit_delay = 600;	/* unit delay in usecs (space bit) */

#define SHORT_WAIT 1000L
#define LONG_WAIT  3000L
#define THRESHOLD  500L

long wait_change(long max) {
   int last;
   long i;

   last = inportb(STATUS);
   for (i = 0; i < max; ++i) {
      if (last != inportb(STATUS)) return i;
   }
   return -1L;
}

static int console_break = 0;

static int break_handler(
void
){
   ++console_break;
   return(1);
}

static char queue[20000];
static char *pqueue = queue;

static void read_bits(void) {
#  define MAX_BITS 200
   int i, num;
   long wait, stat, stats[MAX_BITS];

   num = 0;
   wait = LONG_WAIT;
   while(1) {
      stat = wait_change(wait);
      wait = SHORT_WAIT;
      if (stat < 0) break;
      stats[num++] = stat;
      if (num >= MAX_BITS) break;
   }
   for (i = 1; i < num; i += 2) {
      *pqueue++ = (stats[i] > THRESHOLD) ? '1' : '0';
   }
   *pqueue++ = '\n';
}

void int_processed(void)
/* signals 8259 in PC that interrupt has been processed */
{
   outportb(0x20, 0x20);
}

void interrupt far intserv(void)
/* This is written by the user.  Note that the source of the interrupt
/* must be cleared and then the PC 8259 cleared (int_processed).
/* must be included in this function.
/*******/
{
   disable();
   read_bits();
   int_processed();
   int_occurred=TRUE;
   enable();
}

static int int_serv_opened = 0;

void open_intserv(void)
/* enables IRQ7 interrupt.  On interrupt (low on /ACK) jumps to intserv.
/* all interrupts disabled during this function; enabled on exit.
/*******/
{
   int int_mask;

   int_serv_opened = 1;
   disable();  /* disable all ints */
   oldfunc=getvect(intlev);  /* save any old vector */
   setvect(intlev, intserv);  /* set up for new int serv */
   int_mask=inportb(0x21);    /* 1101 1111 */
   outportb(0x21, int_mask & ~0x80);  /* set bit 7 to zero */
                              /* -leave others alone */
   enable();
}

void close_intserv(void)
/* disables IRQ7 interrupt */
{
   int int_mask;

   if (int_serv_opened) {
      disable();
      setvect(intlev, oldfunc);
      int_mask=inportb(0x21) | 0x80;  /* bit 7 to one */
      outportb(0x21,int_mask);
      enable();
      int_serv_opened = 0;
   }
}

long factor = 1600;

static int delay_usecs(long usecs) {
	long i, num;
	int data;

	num = (usecs * 1000L) / factor;
	for (i = 0; i < num; ++i) {
		data = inportb(DATA);
	}
	return data;
}

#define PP_OUT_BIT	7	/* PC parallel port output bit */

static void setbit(int ch) {
	if (ch & 1) {
		outportb(DATA, 0 << PP_OUT_BIT);
	} else {
		outportb(DATA, 1 << PP_OUT_BIT);
	}
}

void send_command(char *p)
{
	int ch;

	/* start bit */
	setbit(0);
	delay_usecs(4 * unit_delay);
		setbit(1);
		delay_usecs(unit_delay);

	while ((ch = *p++) != 0) {
		setbit(0);
		delay_usecs(unit_delay);
		if (ch == '1') {
			delay_usecs(unit_delay);
		}
		setbit(1);
		delay_usecs(unit_delay);
	}
	setbit(1);
}

void usage(char *program)
{
	fprintf(stderr,
"usage: %s [options...] command\n"
"options:\n"
"   -d        - debug mode\n"
"   -w        - wait for multiple responses\n"
"   -c        - calibrate delay\n"
"   -h        - show all commands\n"
"   -t msecs  - return after \"msecs\" of bus inactivity\n"
"   -p nnn    - use player \"nnn\" (default = 1)\n"
"where \"command\" is of the form \"dddddddd\" (d=0/1)\n"
	"", program);
	exit(1);
}

int main(int argc, char **argv)
{
	long delta_msecs;
	long step_msecs = 200L;
	char *p, *q, *program;
	struct timeb t1, t2;  
	char response[256];

	program = *argv;
	if ((p = strrchr(program, '\\')) != 0) program = p + 1;
	if ((p = strrchr(program, '.')) != 0) *p = '\0';
	--argc; ++argv;

	while (argc && **argv == '-') {
		if (!strcmp(*argv, "-d")) {
			debug = 1;
			--argc; ++argv;
		} else if (!strcmp(*argv, "-h")) {
			text2scode(-1, argc, argv);
			exit(1);
		} else if (!strcmp(*argv, "-c")) {
			calibrate = 1;
			--argc; ++argv;
		} else if (!strcmp(*argv, "-w")) {
			exit_on_first_response = 0;
			--argc; ++argv;
		} else if (!strcmp(*argv, "-t")) {
			--argc; ++argv;
			if (!argc) usage(program);
			timeout_msec = strtol(*argv, NULL, 0);
			if (debug)
				printf("timeout: %ld msecs\n", timeout_msec);
			--argc; ++argv;
		} else if (!strcmp(*argv, "-p")) {
			--argc; ++argv;
			if (!argc) usage(program);
			player = atoi(*argv);
			if (debug)
				printf("player: %d\n", player);
			--argc; ++argv;
		} else {
			fprintf(stderr, "%s: unknown option (%s)!\n",
				program, *argv);
			exit(1);
		}
	}

	if (calibrate) {
		long delay = 1000000L;

		factor = 1000L;
		printf("calibrating...\n");
		ftime(&t1);  /* get start time */  
		delay_usecs(delay);
		ftime(&t2);  /* get end time */  
		delta_msecs = (t2.time * 1000L + t2.millitm) -
		        (t1.time * 1000L + t1.millitm);
		factor =  ((delta_msecs * 1000.0) / delay) * 1000L;
		printf("factor = %g (%ld msec, %ld loops)\n",
			(double)factor, delta_msecs, delay);
		exit(0);
	}

        if (!argc) usage(program);

	ctrlbrk(break_handler) ;    /* install our own ^C/BREAK handler */
	/* set bit 4 on control port (irq disable) to logic zero */
	outportb(CONTROL, inportb(CONTROL) & ~0x10);
	open_intserv();

	p = *argv;
	if (*p != '0' && *p != '1') {
		q = text2scode(player, argc, argv);
		if (*q == '\0') {
			fprintf(stderr, "%s: bad command (%s)! (Use -h for list)\n",
				program, p);
			exit(1);
		} else {
			p = q;
		}
	}
	if (debug) printf("command: %s\n", p);

	disable();	/* disable interrupts */
	send_command(p);
	/* set bit 4 on control port (irq enable) to logic one */
	outportb(CONTROL, inportb(CONTROL) | 0x10);
	enable();	/* re-enable interrupts */

	p = queue;
	q = response;
	delta_msecs = 0;	/* start timeout timer */
	while(!console_break) {
		int ch, len;
		char *text;

		if (debug || timeout_msec == 0) {
			if (kbhit()) break;
		}
		if (*p) {
			ch = *p++;
			len = q - response;
			delta_msecs = 0;	/* re-start timeout timer */
			if (ch == '\n' || len > sizeof(response) - 2) {
				*q++ = '\0';
				q = response;
				text = rcode2text(response);
				if (debug) {
					printf("%s -> %s\n", response, text);
				} else {
					if (*response) {
						printf("%s\n", (*text) ?
							text : response);
					}
				}
				if (len > 1 && exit_on_first_response) {
					break;
				}
			} else {
				*q++ = ch;
				continue;
			}
		} else {
			if (timeout_msec) {
				if (delta_msecs > timeout_msec) {
					if (debug) printf("timeout...\n");
					break;
				}
			}
		}
		delay_usecs(step_msecs * 1000L);	/* wait a bit */
		delta_msecs += step_msecs;
		/***printf("%ld > %ld?\n", delta_msecs, timeout_msec);***/
	}
	close_intserv();
	fflush(stdout);
	return 0;
}

/************************************************************************
  ::::::::::::::
  cd200sls.cde
  ::::::::::::::
  # Sony S-link commands for the cdp-cx200

  desc=Sony CDP-CX200 Send
  name=cdslinks

  type=SLINK

  #Device ID's:

  prefix=10010000	#1
  prefix=10010001	#2
  prefix=10010010	#3
  suffix=

  # the message codes:
  # letters in the codes indicate to the software that this is a variable region

  d:test # this is a test code
  	 # you can test anything with this code by putting hex in brackets
  	 # e.g. test[500304] will play disc 3 track 4

  00000000:play
  00000001:stop
  00000010:pause
  00000011:pause_toggle
  00001000:>>|
  00001001:|<<
  00001111:mode # causes player mode to be reported
  00010000:>>> # ffwd until play command
  00010001:<<< # frew until play command
  00010010:>> # fwd until play command
  00010011:<< # rew until play command
  00011111:unk1 #???
  00100000:remote_off # this stops play, disables the remote control,
  		        # front panel and display so that a remote device
                      # (ie my software) isn't interfered with.
  00100001:remote_on # enable user control
  00100010:player_type # causes player to report type
  00100011:unk2 #???
  00100100:unk3 #???
  00100101:timecode_on # starts second by second timecode reporting (play mode)
  				# reports timecode once if paused
  00100110:timecode_off # stop reporting
  00101110:power_on
  00101111:power_off
  01000000:unk4 #??
  01000001:unk5 #??
  01000100dddddddd:query_disc # disc # must be current disc gives 14 if bad
  					# reports disc info
  01000101ddddddddtttttttt:query_track # disc # must be current disc gives 15 if bad
  					#reports track info
  01010000ddddddddtttttttt:play_dt # play [disk id] [bcd track]
  			# plays requested track
  01010001ddddddddtttttttt:pause_dt # pause at [disk id] [bcd track]
  			# pauses requested track
  01011110rrrrrrrr:fade_down # r = fade rate 0-fast FF-slow
  			# fades volume to zero over specified time
  01011111rrrrrrrr:fade_up # r = fade rate 0-fast FF-slow
  			# ramps volume up over specified time


************************************************************************
   ::::::::::::::
   cd200slr.cde
   ::::::::::::::
   # Sony S-link responses for the cdp-cx200

   desc=Sony CDP-CX200 Receive
   name=cdslinkr

   type=SLINK

   prefix=10011000	#1
   prefix=10011001	#2
   prefix=10011010	#3
   suffix=

   # the message codes:
   # letters in the codes indicate to the software that this is a variable region

   00000000:play
   00000001:stop
   00000010:pause
   00000101:ignored # (displayed when no disc at requested loaction)
   00000110:unloading
   00001000:ready
   00001100:30s_to_eot
   00001110:already_did_that
   00001111:error
   00010100:invalid_disc # used in disc info query
   00010101:invalid_disc_or_track # used in track info query
   00011000:door_open
   00101110:power_on
   00101111:power_off

   01010000ddddddddttttttttmmmmmmmmssssssss:playing # [disk id] [bcd track] [bcd min] [bcd sec]
   01010001iiiiiiiittttttttmmmmmmmmssssssss:time # [bcd index] [bcd track] [bcd min] [bcd sec]
   01010010dddddddd:displaying_disc	# [disk id]
   01010011dddddddd:no_disc # [disc id] (displayed when no disc at requested loaction)
   01010100dddddddd:goto_disc	# [disk id]
   01011000dddddddd:now_at_disc	# [disk id]
   01100000ddddddddiiiiiiiittttttttmmmmmmmmssssssssffffffff:disc_info # [disk id] [bcd indexes?] [bcd tracks] [bcd total min] [bcd sec] [bcd frames?]
   01100010ddddddddttttttttmmmmmmmmssssssss:track_info # [disk id]  [bcd track] [bcd  min] [bcd  sec]
   01100001ssssssssxxxxxxxx:player_type # [capacity (in disk id format)] [unk]
   01110000ooooooopKArRooPSxoooooooddddddddtttttttt:mode # p=playing(1) K=scanning(0)/discs known (1) A=1 disc(0)/all discs(1) r=repeat1 (1) R=repeat all(1) P= program(1) S=shuffle (1)
   10000011:door_close

************************************************************************/

char *rplayers[] = {
	"10011000",	/* player #0 */
	"10011001",	/* player #1 */
	"10011010",	/* player #2 */
	NULL
};

struct rcodes {
	char *name;
	char *code;
	int type;
	int disc_id;
} rcode_table[] = {
	{ "play",			"00000000", 0, 0 },
	{ "stop",			"00000001", 0, 0 },
	{ "pause",			"00000010", 0, 0 },
	{ "ignored",			"00000101", 0, 0 },
	{ "unloading",			"00000110", 0, 0 },
	{ "ready",			"00001000", 0, 0 },
	{ "30s_to_eot",			"00001100", 0, 0 },
	{ "already_did_that",		"00001110", 0, 0 },
	{ "error",			"00001111", 0, 0 },
	{ "invalid_disc",		"00010100", 0, 0 },
	{ "invalid_disc_or_track",	"00010101", 0, 0 },
	{ "door_open",			"00011000", 0, 0 },
	{ "power_on",			"00101110", 0, 0 },
	{ "power_off",			"00101111", 0, 0 },
	{ "playing",			"01010000", 1, 1 },
	{ "time",			"01010001", 2, 0 },
	{ "displaying_disc",		"01010010", 3, 1 },
	{ "no_disc",			"01010011", 4, 1 },
	{ "goto_disc",			"01010100", 5, 1 },
	{ "now_at_disc",		"01011000", 6, 1 },
	{ "disc_info",			"01100000", 7, 1 },
	{ "track_info",			"01100010", 8, 1 },
	{ "player_type",		"01100001", 9, 0 },
	{ "mode",			"01110000", 10,0 },
	{ "door_close",			"10000011", 0, 0 },
	{ NULL, NULL, -1, -1 }
};

/************************************************************************
 <Disc#>: 1 byte can have 256 unique numbers.  Sony ignores disc 00
          on the 200 Disc Players.  Disc 100 on the 100 Disc Players
          is represented as 00 hex.  Also, their "hacked" format for
          200 discs (their format for the 100 disc players is simple,
          just BCD) allows a total of 201 possible discs.  This
          explains Discs 100-200 being equal to the
          hex value - 54(base 10) while Discs 1-99 are in BCD.  The
          algorithm to take the byte given for disc# and printing it
          on a computer screen in base 10 is as follows:

          //Comment: (hb1 = 4MSB and hb2 = 4LSB of the byte)
          if (hb1 > 9)
             print("Disc #" + (16*hb1 + hb2 - 54))
          else
             print("Disc #" + (10*hb1 + hb2))

************************************************************************/

int bits2int(char *s, int bits)
{
	int i, num = 0;

	for (i = 0; i < bits; ++i, ++s) {
		if (!*s) break;
		switch(*s) {
		case '0':
			num = (num << 1);
			break;
		case '1':
			num = (num << 1) + 1;
			break;
		default:
			goto done;
		}
	}
done:
	if ((num >> 4) > 9) {
		return (16 * (num >> 4)) + (num & 0xf) - 54;
	} else {
		return (10 * (num >> 4)) + (num & 0xf);
	}
}

char *int2bits(int num)
{
	int i, hi, lo, data;
	static char *p, code[16];

	if (num >= 100) {
		data = num + 54;
	} else {
		hi = num / 10;
		lo = num % 10;
		data = (hi << 4) | lo;
	}

	p = code;
	for (i = 0; i < 8; ++i) {
		*p++ = (data & 0x80) ? '1' : '0';
		data = data << 1;
	}

	return code;
}

char *rcode2text(char *code)
{
	int i, disc, track, mins, secs, index, frame;
	char *s, *p, *q;
	static char text[256];

	p = code;
	q = text;

	/* find CD player (prefix) */
	for (i = 0; rplayers[i]; ++i) {
		if (!strncmp(p, rplayers[i], 8)) break;
	}
	if (!rplayers[i]) {
		return "";
	}
	sprintf(q, "player %d:", i);
	q = text + strlen(text);
	p += 8;

	/* find response code */
	for (i = 0; rcode_table[i].name; ++i) {
		if (!strncmp(p, rcode_table[i].code, 8)) break;
	}
	if (rcode_table[i].name) {
		s = rcode_table[i].name;
	} else {
		return "";
	}
	sprintf(q, " %s", s);
	q = text + strlen(text);
	p += 8;

	if (rcode_table[i].disc_id) {
		sprintf(q, " disc %d", bits2int(p, 8));
		q = text + strlen(text);
		p += 8;
	}

	switch(rcode_table[i].type) {
	/* ttttttttmmmmmmmmssssssss:playing
	   [disk id] [bcd track] [bcd min] [bcd sec] */
	case 1:
		track = bits2int(p, 8);
		mins = bits2int(p + 8, 8);
		secs = bits2int(p + 16, 8);
		sprintf(q, " trk %d len %02d:%02d", track, mins, secs);
		break;

	/* ttttttttmmmmmmmmssssssss:time
	   [bcd index] [bcd track] [bcd min] [bcd sec] */
	case 2:
		track = bits2int(p, 8);
		mins = bits2int(p + 8, 8);
		secs = bits2int(p + 16, 8);
		sprintf(q, " trk %d len %02d:%02d", track, mins, secs);
		break;

	/* :displaying_disc	# [disk id] */
	case 3:
		break;

	/* :no_disc # [disc id] (when no disc at requested loaction) */
	case 4:
		break;

	/* :goto_disc	# [disk id] */
	case 5:
		break;

	/* :now_at_disc	# [disk id] */
	case 6:
		break;

	/* iiiiiiiittttttttmmmmmmmmssssssssffffffff:disc_info
	   [disk id] [bcd indexes?] [bcd tracks] [bcd total min] [bcd sec]
	   [bcd frames?] */
	case 7:
		index = bits2int(p, 8); p += 8;
		track = bits2int(p, 8); p += 8;
		mins = bits2int(p, 8); p += 8;
		secs = bits2int(p, 8); p += 8;
		frame = bits2int(p, 8); p += 8;
		sprintf(q, " index %d trk %d len %02d:%02d frame %d",
			index, track, mins, secs, frame);
		break;

	/* ttttttttmmmmmmmmssssssss:track_info
	   [disk id]  [bcd track] [bcd  min] [bcd  sec] */
	case 8:
		track = bits2int(p, 8);
		mins = bits2int(p + 8, 8);
		secs = bits2int(p + 16, 8);
		sprintf(q, " trk %d len %02d:%02d", track, mins, secs);
		break;

	/* ssssssssxxxxxxxx:player_type
	   [capacity (in disk id format)] [unk] */
	case 9:
		sprintf(q, " %s", p);
		break;

	case 10:
	/* ooooooopKArRooPSxoooooooddddddddtttttttt:mode
           p=playing(1) K=scanning(0)/discs known (1) A=1 disc(0)/all discs(1)
           r=repeat1 (1) R=repeat all(1) P= program(1) S=shuffle (1) */
		disc = bits2int(p + 24, 8);
		track = bits2int(p + 32, 8);
		sprintf(q, " disc %d track %d%s%s%s%s%s%s%s",
			disc, track, 
			(*(p+7)== '1') ? " playing" : "",
			(*(p+8)== '1') ? " discs_known": " scanning",
			(*(p+9)== '1') ? " all_discs" : " one_disc",
			(*(p+10)== '1') ? " repeat_one" : "",
			(*(p+11)== '1') ? " repeat_all" : "",
			(*(p+14)== '1') ? " program" : "",
			(*(p+15)== '1') ? " shuffle" : "");
		break;
	}

	return text;
}

char *splayers[] = {
	"10010000",	/* player #0 */
	"10010001",	/* player #1 */
	"10010010",	/* player #2 */
	NULL
};

struct scodes {
	char *name;
	char *code;
	int parms;
} scode_table[] = {
	{ "play",		"00000000", 0 },
	{ "stop",		"00000001", 0 },
	{ "pause",		"00000010", 0 },
	{ "pause_toggle",	"00000011", 0 },
	{ "next_track",	"00001000", 0 },
	{ "back_track",	"00001001", 0 },
	{ "mode",		"00001111", 0 },
	{ "ffwd",		"00010000", 0 },
	{ "frew",		"00010001", 0 },
	{ "fwd",		"00010010", 0 },
	{ "rew",		"00010011", 0 },
	{ "unk1",		"00011111", 0 },
	{ "remote_off",	"00100000", 0 },
	{ "remote_on",	"00100001", 0 },
	{ "player_type",	"00100010", 0 },
	{ "unk2",		"00100011", 0 },
	{ "unk3",		"00100100", 0 },
	{ "timecode_on",	"00100101", 0 },
	{ "timecode_off",	"00100110", 0 },
	{ "power_on",	"00101110", 0 },
	{ "power_off",	"00101111", 0 },
	{ "unk4",		"01000000", 0 },
	{ "unk5",		"01000001", 0 },
	{ "query_disc",	"01000100", 1 },
	{ "query_track",	"01000101", 2 },
	{ "play_dt",	"01010000", 2 },
	{ "pause_dt",	"01010001", 2 },
	{ "fade_down",	"01011110", 1 },
	{ "fade_up",	"01011111", 1 },
	{ NULL, NULL, -1 }
};

char *text2scode(int player, int argc, char **argv)
{
	int i, num, parms;
	char *s, *p, *q;
	static char code[256];

	if (player == -1) {
		printf("commands:\n");
		for (i = 0; scode_table[i].name; ++i) {
			parms = scode_table[i].parms;
			printf("   %s", scode_table[i].name);
			while (parms--) {
				printf(" <parm>");
			}
			printf("\n");
		}
		return "";
	}

	q = code;
	/* find CD player (prefix) */
	for (i = 0; splayers[i]; ++i) {
		if ((player - 1) == i) break;
	}
	if (!splayers[i]) {
		return "";
	}
	strcpy(q, splayers[i]);
	q = code + strlen(code);

	/* find command */
	p = argv[0];
	for (i = 0; scode_table[i].name; ++i) {
		if (!strcmp(p, scode_table[i].name)) break;
	}
	if (scode_table[i].name) {
		strcpy(q, scode_table[i].code);
		q = code + strlen(code);
	} else {
		return "";
	}

	parms = scode_table[i].parms;
	if ((argc - 1) == parms) {
		for (i = 0; i < parms; ++i) {
			num = atoi(argv[i+1]);
			strcpy(q, int2bits(num));
			q = code + strlen(code);
		}
	} else {
		return "";
	}

	return code;
}
