# senddata.pl - send data to PicoWeb.
#
# no batching - just sends one datagram for each record from STDIN.
#

$debug = 0;
$wait = 0;
$ir = 0;
$msecs = 10000;
$player = 0;
$cmd_port = $ENV{"CMDPORT"} || 1999;

$subs = 'subs.pl';
if (-f  $subs) {
    require $subs
} else {
    $dir = "$0\n";
    $dir =~ s/\\/\//g;
    $dir =~ s/\/[^\/]+$//;
    require "$dir/$subs";
}

select((select(STDOUT), $| = 1)[0]);    # unbuffer output
select((select(STDERR), $| = 1)[0]);

#       name                  code    ,no. params
$cmd2scode{"play"}         = "00000000,0";
$cmd2scode{"stop"}         = "00000001,0";
$cmd2scode{"pause"}        = "00000010,0";
$cmd2scode{"pause_toggle"} = "00000011,0";
$cmd2scode{"next_track"}   = "00001000,0";
$cmd2scode{"back_track"}   = "00001001,0";
$cmd2scode{"mode"}         = "00001111,0";
$cmd2scode{"ffwd"}         = "00010000,0";
$cmd2scode{"frew"}         = "00010001,0";
$cmd2scode{"fwd"}          = "00010010,0";
$cmd2scode{"rew"}          = "00010011,0";
$cmd2scode{"unk1"}         = "00011111,0";
$cmd2scode{"remote_off"}   = "00100000,0";
$cmd2scode{"remote_on"}    = "00100001,0";
$cmd2scode{"player_type"}  = "00100010,0";
$cmd2scode{"unk2"}         = "00100011,0";
$cmd2scode{"unk3"}         = "00100100,0";
$cmd2scode{"timecode_on"}  = "00100101,0";
$cmd2scode{"timecode_off"} = "00100110,0";
$cmd2scode{"power_on"}     = "00101110,0";
$cmd2scode{"power_off"}    = "00101111,0";
$cmd2scode{"unk4"}         = "01000000,0";
$cmd2scode{"unk5"}         = "01000001,0";
$cmd2scode{"query_disc"}   = "01000100,1";
$cmd2scode{"query_track"}  = "01000101,2";
$cmd2scode{"play_dt"}      = "01010000,2";
$cmd2scode{"pause_dt"}     = "01010001,2";
$cmd2scode{"fade_down"}    = "01011110,1";
$cmd2scode{"fade_up"}      = "01011111,1";

#   name =                           code, type, disc_id
$cmd2rcode[&bits2int("00000000")] = "play,0,0";
$cmd2rcode[&bits2int("00000001")] = "stop,0,0";
$cmd2rcode[&bits2int("00000010")] = "pause,0,0";
$cmd2rcode[&bits2int("00000101")] = "ignored,0,0";
$cmd2rcode[&bits2int("00000110")] = "unloading,0,0";
$cmd2rcode[&bits2int("00001000")] = "ready,0,0";
$cmd2rcode[&bits2int("00001100")] = "30s_to_eot,0,0";
$cmd2rcode[&bits2int("00001110")] = "already_did_that,0,0";
$cmd2rcode[&bits2int("00001111")] = "error,0,0";
$cmd2rcode[&bits2int("00010100")] = "invalid_disc,0,0";
$cmd2rcode[&bits2int("00010101")] = "invalid_disc_or_track,0,0";
$cmd2rcode[&bits2int("00011000")] = "door_open,0,0";
$cmd2rcode[&bits2int("00101110")] = "power_on,0,0";
$cmd2rcode[&bits2int("00101111")] = "power_off,0,0";
$cmd2rcode[&bits2int("01010000")] = "playing,1,1";
$cmd2rcode[&bits2int("01010001")] = "time,2,0";
$cmd2rcode[&bits2int("01010010")] = "displaying_disc,3,1";
$cmd2rcode[&bits2int("01010011")] = "no_disc,4,1";
$cmd2rcode[&bits2int("01010100")] = "goto_disc,5,1";
$cmd2rcode[&bits2int("01011000")] = "now_at_disc,6,1";
$cmd2rcode[&bits2int("01100000")] = "disc_info,7,1";
$cmd2rcode[&bits2int("01100010")] = "track_info,8,1";
$cmd2rcode[&bits2int("01100001")] = "player_type,9,0";
$cmd2rcode[&bits2int("01110000")] = "mode,10,0";
$cmd2rcode[&bits2int("10000011")] = "door_close,0,0";

#        key      codes    name      description   extra
$key2code{'!'} = "1010100, power,    power,        *";
$key2code{'c'} = "1011100, continue, continue,     *";
$key2code{'s'} = "1010110, shuffle,  shuffle,      *";
$key2code{'P'} = "1111100, program,  program,      *";
$key2code{'r'} = "0011010, repeat,   repeat,       *";
$key2code{'+'} = "0111110, skip+,    next disk,    *";
$key2code{'-'} = "1011110, skip-,    prev disk,    *";
$key2code{'p'} = "0100110, play,     play,         *";
$key2code{'"'} = "1001110, pause,    pause,        *";
$key2code{'l'} = "0000110, last,     last select,  *";
$key2code{'n'} = "1000110, next,     next select,  *";
$key2code{'.'} = "0001110, stop,     stop,         *";
$key2code{'<'} = "1100110, reverse,  reverse,      *";
$key2code{'>'} = "0010110, forward,  fast forward, *";
$key2code{'d'} = "0101001, disc,     select disc,  *";
$key2code{'t'} = "1101001, track,    select track, *";
$key2code{'g'} = "0011001, group,    select group, *";
$key2code{'1'} = "0000000, 1,        1,            *";
$key2code{'2'} = "1000000, 2,        2,            *";
$key2code{'3'} = "0100000, 3,        3,            *";
$key2code{'4'} = "1100000, 4,        4,            *";
$key2code{'5'} = "0010000, 5,        5,            *";
$key2code{'6'} = "1010000, 6,        6,            *";
$key2code{'7'} = "0110000, 7,        7,            *";
$key2code{'8'} = "1110000, 8,        8,            *";
$key2code{'9'} = "0001000, 9,        9,            *";
$key2code{'0'} = "0000010, 10/0,     10/0,         *";
$key2code{'e'} = "1101000, enter,    enter,        *";
$key2code{'m'} = "1001011, memo,     memo,         *";
$key2code{'D'} = "1100100, vol-,     volume down,  *";
$key2code{'U'} = "0100100, vol+,     volume up,    *";
$key2code{']'} = "0000011, jog+,     jog up,       sleep=60";
$key2code{'['} = "1000011, jog-,     jog down,     sleep=60";
$key2code{'}'} = "0100011, jog+!,    double jog up,sleep=60";
$key2code{')'} = "0010011, jog+!!,   quad jog up,  sleep=60";

$ir_suffix[0] = "10001";
$ir_suffix[1] = "10011100";
$ir_suffix[2] = "10001010";

sub getircode {
    local($ch) = @_;
    local($n, %F, $codes, $s);

    $s = $key2code{$ch};
    if ($s eq '') {
        print STDERR "unknown IR key '$ch'!\n";
        exit 1;
    }
    $n = @F = split(',', $s);
    $codes = $F[0];
    return $codes;
}

sub text2ircode {
    local($buf) = @_;
    local($player, $arg, $ch);

    $num_lines = 0;
    while (1) {
        $arg = $ARGV[$arg_ix++];
        return $s if ($arg eq '');
        while ($arg ne '') {
            $ch = substr($arg, 0, 1);
            $arg = substr($arg, 1);
            $lines[$num_lines++] = &getircode($ch) . $ir_suffix[$player];
        }
    }
}

sub text2rcode {
    local($buf) = @_;
    local($player, @p, $p, $ix, $type, $id, $s, $cmd, $F, @F, $n);

    $ix = 0;
    $player = unpack('C', substr($buf, $ix++, 1));
    $player = $player - 0x98;
    print "player $player: ";
    $cmd = unpack('C', substr($buf, $ix++, 1));
    $s = $cmd2rcode[$cmd];
    if ($s eq '') {
        printf(STDERR "unknown command 0x%02x!\n", $cmd);
        exit 1;
    }
    $n = @F = split(',', $s);

    $s = $F[0];
    print "$s";

    $type = $F[1];
    $id = $F[2];

    if ($id) {
        $s = substr($buf, $ix++, 1);
        $id = unpack('C', $s);
        $id = &hex2int($id);
        print " disc $id";
    }

    $n = 0;
    while (1) {
        $s = substr($buf, $ix++, 1);
        last if ($s eq '');
        $s = unpack('C', $s);
        $p[$n++] = &hex2int($s);
    }

    if (     $type == 1) {
        printf(" trk %d len %02d:%02d", $p[0], $p[1], $p[2]);
    } elsif ($type == 2) {
        printf(" trk %d len %02d:%02d", $p[0], $p[1], $p[2]);
    } elsif ($type == 3) {
        ;
    } elsif ($type == 4) {
        ;
    } elsif ($type == 5) {
        ;
    } elsif ($type == 6) {
        ;
    } elsif ($type == 7) {
        printf("  index %d trk %d len %02d:%02d frame %d",
               $p[0], $p[1], $p[2], $p[3], $p[4]);
    } elsif ($type == 8) {
        printf(" trk %d len %02d:%02d", $p[0], $p[1], $p[2]);
    } elsif ($type == 9) {
        printf(" %d %d", $p[0], $p[1]); 
    #            1          2          3
    # 01234567 890123456 7890123 45678901 2
    # ooooooop KArRooPSx ooooooo dddddddd tttttttt:mode
    #  p=playing(1) K=scanning(0)/discs known (1) A=1 disc(0)/all discs(1)
    #  r=repeat1 (1) R=repeat all(1) P= program(1) S=shuffle (1) */
    } elsif ($type == 10) {
        printf(" disc %d track %d%s%s%s%s%s%s%s", $p[3], $p[4],
                ($p[0] & 0x01) ? " playing" : "",
                ($p[1] & 0x80) ? " discs_known" : " scanning",
                ($p[1] & 0x40) ? " all_discs" : " one_disc",
                ($p[1] & 0x20) ? " repeat_one" : "",
                ($p[1] & 0x10) ? " repeat_all" : "",
                ($p[1] & 0x04) ? " program" : "",
                ($p[1] & 0x02) ? " shuffle" : ""
        );
    }
    print "\n";
}

sub text2scode {
    local($ss, $i, $parm, $s, $cmd, $F, @F, $n);

    $cmd = $ARGV[$arg_ix++];
    return $cmd if ($cmd =~ /^[01]+$/);

    $s = '';
    $s .= &int2bits($player + 0x90);
    $ss .= $cmd2scode{$cmd};
    if ($ss eq '') {
        print STDERR "unknown command \"$cmd\"!\n";
        exit 1;
    }
    $s .= $ss;
    $n = @F = split(',', $s);
    $s = $F[0];
    $n = $F[1];
    while ($n--) {
        $parm = $ARGV[$arg_ix++];
        if ($parm eq '') {
            print STDERR "missing parameter for \"$cmd\"!\n";
            exit 1;
        }
        if ($parm >= 100) {
            $parm = $parm + 54;
        } else {
            $parm = (($parm / 10) << 4) | ($parm % 10);
        }
        $s .= &int2bits($parm);
    }
    return $s;
}

sub bits2int {
    local($bits) = @_;
    local($n, $ch, $i);

    $n = 0;
    while (1) {
        $ch = substr($bits, $i++, 1);
        last if ($ch eq '');
        $n = $n << 1;
        $n |= 1 if ($ch eq '1');
    }
    return $n;
}

sub hex2int {
    local($num) = @_;

    if (($num >> 4) > 9) {
        return (16 * ($num >> 4)) + ($num & 0xf) - 54;
    } else {
        return (10 * ($num >> 4)) + ($num & 0xf);
    }
}

sub int2bits {
    local($val) = @_;
    local($i, $s);

    $s = '';
    for ($i = 0; $i < 8; ++$i) {
        $s .= ($val & 0x80) ? '1' : '0';
        $val = $val << 1;
    }
    return $s;
}

sub usage {
    print 
    "usage: %s [options...] command\n".
    "options:\n".
    "   -d            - debug mode\n".
    "   -w            - wait for multiple responses\n".
    "   -h            - show all commands\n".
    "   -ir           - IR send mode\n".
    "   -t msecs      - return after \"msecs\" of bus inactivity\n".
    "   -p nnn        - use player \"nnn\" (default = 1)\n".
    "   -ip x.y.z.w[:p] - set IP address [and port] of PicoWeb\n".
    "where \"command\" is of the form \"dddddddd\" (d=0/1)\n";
    exit(1);
}

&usage if ($ARGV[0] eq '');

#
#   Parse passed parameters
#
while ($ARGV[0] =~ /^-/) {
    $_ = shift;
    if (/^-d$/) {
        $debug = 1;
    } elsif (/^-w$/) {
        $wait = 1;
    } elsif (/^-ir$/) {
        $ir = 1;
    } elsif (/^-h/) {
        &usage;
    } elsif (/^-ip$/) {
        $addr = shift;
        if ($addr =~ /^(.*)[:](.*)$/) {
            $addr = $1;
            $cmd_port = $2;
        }
    } elsif (/^-t$/) {
        $msecs = shift;
        $msecs += 0;
        $msecs = 2000 if ($msecs > 2000);
    } elsif (/^-p$/) {
        $player = shift;
        $player += 0;
    } else {
        die "$ego: Unrecognized switch: $_ (-h for help)\n";
    }
}

if ($addr eq '') {
    die "missing UDP host IP address!\n";
}

if ($cmd_port == 0) {
    die "bad UDP port!\n";
}

&usage if ($ARGV[0] eq '');

$arg_ix = 0;
if ($ir) {
    $line = &text2ircode;
} else {
    $line = &text2scode;
}

$AF_INET = 2;
$SOCK_STREAM = 1;
$SOCK_DGRAM = 2;
$INADDR_ANY = pack("N",0);
$SOCKADDR = "S n a4 x8";
($name,$aliases,$IPPROTO_UDP) = getprotobyname('udp');

$| = 1;

$listenaddr = pack($SOCKADDR,$AF_INET,0,$INADDR_ANY);
#
# sockaddr representing the picoweb we're talking to.
#
$sendaddr = pack($SOCKADDR,$AF_INET,$cmd_port,&inet_addr($addr));

socket(S,$AF_INET,$SOCK_DGRAM,$IPPROTO_UDP) || die "socket: $!";
bind(S,$listenaddr) || die "bind: $!";

##$mysockaddr = getsockname(S);
##($family, $port, $myaddr) = unpack($SOCKADDR,$mysockaddr);
print STDERR "sending to $addr:$cmd_port ($line)\n" if ($debug);

if ($ir) {
    for ($i = 0; $i < $num_lines; ++$i) {
        &sendline($lines[$i]);
    }
} else {
    &sendline($line);
}

exit 0;

sub sendline {
    local($line) = @_;
    local($buf, $s, $num, $i, $len);

    $buf = '';
    if ($ir) {
        $buf .= pack('C', length($line));
    } else {
        $buf .= pack('C', 0);
    }
    for ($i = 0; $i < length($line); $i += 8) {
        $s = substr($line, $i, 8);
        $s .= '0' while (length($s) != 8);
        $num = &bits2int($s);
        $buf .= pack('C', $num);
    }
    if ($debug) {
        $len = length($buf);
        print STDERR "sending $len bytes:" if ($debug);
        for ($i = 0; $i < $len; ++$i) {
            $num = unpack('C', substr($buf, $i, 1));
            printf(STDERR " %02x", $num) if ($debug);
        }
        print STDERR "\n";
    }

    send(S, $buf, 0, $sendaddr);
    &wait_ack if ($wait);
}

sub wait_ack {
    undef $rin;
    $now = time;
    print STDERR "waiting...$now\n" if ($debug);
    for ($tries = 0; $tries < 90; ++$tries) {
        vec($rin,fileno(S),1) = 1;
        $n = select($rout=$rin,undef,undef,0.1);
        if (vec($rout,fileno(S),1))  {
            $n = recv(S,$buf,1024,0);
            last if (length($buf) == 0);
            $len = length($buf);
            $hdr = unpack("S",$buf);
            $buffer = substr($buf,2);
            if ($debug) {
                printf(STDERR "recv($len bytes): %04x:", $hdr);
                for ($i = 0; $i < length($buffer); ++$i) {
                    printf(STDERR " %02x", unpack('C', substr($buffer, $i, 1)));
                }
                print STDERR "\n";
            }
            $buffer = substr($buffer, 2);
            if ($debug) {
                for ($i = 0; $i < length($buffer); ++$i) {
                    $n = unpack('C', substr($buffer, $i, 1));
                    print STDERR &int2bits($n);
                    printf(STDERR "[%02x]", $n);
                }
                print STDERR "\n";
            }
            &text2rcode($buffer) if ($ir == 0);
            return;
        } else {
            $secs = time - $now;
            ###print STDERR "$secs seconds!\n" if ($debug);
            return if ($secs * 1000 > $msecs);
            print STDERR "." if ($debug);
        }
    }
    print STDERR "\n" if ($debug);
}

