/*
* Program IR.C
*
*/

#include <stdio.h>
#include <stdlib.h>
//#include <bios.h>
#include <dos.h>
#include <string.h>
#include <sys\timeb.h>

#ifdef WIN32
void outportb(int port, int data) {
	_asm mov edx,port;
	_asm mov al,byte ptr data;
	_asm out dx,al;
}

void disable(void) {_asm cli }
void enable(void) {_asm sti }

#define DEF_LOOPS 270
#else
#define DEF_LOOPS 185
#endif

//#define DATA 0x0378
#define DATA 0x03BC
#define STATUS DATA+1
#define CONTROL DATA+2

#define TRUE 1
#define FALSE 0

static int debug = 0;
static int noints = 1;
static int calibrate = 0;
static int player = 1;
static long in_loops = DEF_LOOPS, out_loops = 100000;

/************************************************************************
# @(#) cx100.dat 1.0 95/01/09
# For the Sony CDP-CX100 100 CD changer/player
# John H. DuBois III 
# Modified from data for the RM-D505, which has similar codes

name=Sony CD Player Remote RM-DX100

# Sony remote sensor works well with a transmitter rate of 1600
# transitions/sec

# carrier is the carrier frequency required, in Hz.
# This information may be used by a driver that has access to more than
# one transmitter, or with a system that can set the transmitter frequency.
carrier=40000

# cd player needs power for ~380 mS before it will accept remote commands

# Transmitter codes are given as a string of 1's and 0's in the function code
# table.  The actual IR pulse codes emitted for each 1 and 0 are given by the
# value that "one" and "zero" are set to.  Each character of the values of 
# "one" and "zero" indicates whether the 40 KHz IR transmitter is on for a unit
# of time.  The duration of the unit of time represented by each character of
# the values of "one" and "zero" is given by the value of "cycles".

# The RM-DX100 has a switch to select between CD players 1, 2, and 3.
# The position of the switch changes the suffix.  The default suffix given
# here is for the remote with the switch set to player 1.
# The suffixes for players 2 and 3 are 10011100 and 10001010

component=cd

# These are codes not sent by the remote that produce a change in the display
# of unknown meaning
# 0010011
# 0100011
# 1010011
# 1011000
# 1100011
# 1111000	This one once changed the display to '1 DELETE'...
************************************************************************/

struct ir_codes {
	char *code;
	int key;
	char *word;
	char *label;
	char *vars;
	char *descr;
} ir_code_table[] = {
/*  Code    Key   Word       Label         Vars  Description */
{"1010100", '!', "power",   "power",       "*", "Power on/off"},
{"1011100", 'c', "continue","continue",    "*", "Cont play toggle"},
{"1010110", 's', "shuffle", "shuffle",     "*", "Suffle toggle"},
{"1111100", 'P', "program", "program",     "*", "Program mode"},
{"0011010", 'r', "repeat",  "repeat",      "*", "Repeat toggle"},
{"0111110", '+', "skip+",   "next disk",   "*", "Go to next disk"},
{"1011110", '-', "skip-",   "prev disk",   "*", "Go to prev disk"},
{"0100110", 'p', "play",    "play",        "*", "Play disk"},
{"1001110", '"', "pause",   "pause",       "*", "Pause/Unpause"},
{"0000110", 'l', "last",    "last select", "*", "Start of current selection"},
{"1000110", 'n', "next",    "next select", "*", "Start of next selection"},
{"0001110", '.', "stop",    "stop",        "*", "Stop playing/delete select"},
{"1100110", '<', "reverse", "reverse",     "*", "Reverse search"},
{"0010110", '>', "forward", "fast forward","*", "Forward search"},
{"0101001", 'd', "disc",    "select disc", "*", "Select disc by no. [CAPS]"},
{"1101001", 't', "track",   "select track","*", "Select track by no. [SPACE]"},
{"0011001", 'g', "group",   "select group","*", "Select group by no. [DEL}"},
{"0000000", '1', "1",       "1",           "*", "1: disk/track/group number"},
{"1000000", '2', "2",       "2",           "*", "2: disk/track/group number"},
{"0100000", '3', "3",       "3",           "*", "3: disk/track/group number"},
{"1100000", '4', "4",       "4",           "*", "4: disk/track/group number"},
{"0010000", '5', "5",       "5",           "*", "5: disk/track/group number"},
{"1010000", '6', "6",       "6",           "*", "6: disk/track/group number"},
{"0110000", '7', "7",       "7",           "*", "7: disk/track/group number"},
{"1110000", '8', "8",       "8",           "*", "8: disk/track/group number"},
{"0001000", '9', "9",       "9",           "*", "9: disk/track/group number"},
{"0000010", '0', "10/0",    "10/0",        "*", "0: disk/track/group number"},
{"1101000", 'e', "enter",   "enter",       "*", "number: disk/track/group"},
//{"0001010", 'm', "memo",    "memo",        "*", "memo"},
{"1001011", 'm', "memo",    "memo",        "*", "memo"},
{"1100100", 'D', "vol-",    "volume down", "*", "Line level down"},
{"0100100", 'U', "vol+",    "volume up",   "*", "Line level up"},
{"0000011", ']', "jog+",    "jog up",      "sleep=60", "Jog-next disc"},
{"1000011", '[', "jog-",    "jog down",    "sleep=60,repeat=1", ""},
{"0100011", '}', "jog+!",   "double jog up","sleep=60", ""},
{"0010011", ')', "jog+!!",  "quad jog up", "sleep=60", ""},
{NULL, 0, NULL,  NULL,   NULL, NULL}
};

/************************************************************************/

// prefix and suffix give the standard preamble and postamble, if any,
// that come immediately before and after the function code.
// Prefix and suffix are given in the same representation as function codes.
static char *prefix = "";
static char *suffix = "10001";

// cycles is the number of 40 KHz cycles (25 uS periods) represented by each
// character in the definitions of "one" and "zero".
static int cycles = 24;

// pause is the length of time to wait between code repeats.
// It is given in the units given by the definition of "cycles"
// instead of in mS so that equipment that requires a very short
// pause can be accomodated.
static int pause = 41;

// sleep is the length of time to wait between sending different codes, 
// in milliseconds
// Actual minimum measured at 60 mS
static int ir_sleep = 100;

// repeat is the number of times the code should be sent
static int repeat = 2;

// zero and one describe the waveform used to transmit a zero and one as given
// in the function table.  A '~' represents a period of tranmitter "on" time,
// during which time the emitter will be modulated by a 40 KHz square wave.
// For each '~', <cycles> IR pulses will be transmitted.  A '_' (underscore)
// represents a period of transmitter "off" time.  The period is given by the
// value of cycles.  
static char *zero = "~_";
static char *one = "~~_";

// start and stop give start and stop codes, if any, that cannot be described
// using ones and zeros as used in the function table and so cannot be given
// as prefixes and suffixes.
// start and stop are given in the same representation as zero and one.
// start and stop are the first and last codes transmitted (they are sent
// before and after prefix and suffix, respectively).
static char *start = "~~~~_";
static char *stop = "";

/************************************************************************/

static int console_break = 0;

static int break_handler(
void
){
   ++console_break;
   return(1);
}

static int send_ir(char *codes, long loops, long delay, int noints) {
	long i, num1, num2, l, numl;
	int bit, data, r = 0;
	char *p = codes;

	if (debug)
		printf("send_ir(%s, %ld, %ld, %d)\n",
			codes, loops, delay, noints);

	num1 = delay / 2;
	num2 = delay - num1;

	if (noints) disable();
	for (  ; *p ; ++p) {
		bit = (*p == '~') ? 0x01 : 0x00;
		numl = (*p == 'P') ? (41 * loops) : loops;
		for (l = 0; l < numl; ++l) {
			outportb(DATA, bit);
			for (i = 0; i < num1; ++i) {
				++r;
				///data = inportb(DATA);
			}
			outportb(DATA, 0x00);
			for (i = 0; i < num2; ++i) {
				++r;
				///data = inportb(DATA);
			}
		}
	}
	if (noints) enable();
	return r;
}

static int test_40khz(long loops, long delay) {
	long i, num1, num2;
	int data, r = 0;

	num1 = delay / 2;
	num2 = delay = num2;
	if (noints) disable();
	while (--loops > 0) {
		outportb(DATA, 0x01);
		for (i = 0; i < num1; ++i) {
			++r;
			///data = inportb(DATA);
		}
		outportb(DATA, 0x00);
		for (i = 0; i < num2; ++i) {
			++r;
			///data = inportb(DATA);
		}
	}
	if (noints) enable();
	return r;
}

char *ircode2level(char *code) {
	int r = repeat, i, num, ch;
	static char result[4096];
	char *p;

	*result = '\0';
	while (r-- > 0) {
		strcat(result, start);
		p = code;
		while ((ch = *p++) != '\0') {
			switch(ch) {
			case '0':
				strcat(result, zero);
				break;
			case '1':
				strcat(result, one);
				break;
			}
		}
		strcat(result, stop);
#if 0
		num = (pause + (cycles - 1)) / cycles;
		//num = pause;
		for (i = 0; i < num; ++i) {
			strcat(result, "_");
		}
#else
		strcat(result, "P");
#endif
	}
	return result;
}

int key2ircode(int key) {
	int i;

	/* find key */
	for (i = 0; ir_code_table[i].key; ++i) {
		if (key == ir_code_table[i].key) break;
	}
	if (ir_code_table[i].key) {
		return i;
	} else {
		return -1;
	}
}

void show_ir_table(void) {
	int i;

	for (i = 0; ir_code_table[i].key; ++i) {
		printf(" %c  %-16s  %s\n",
			ir_code_table[i].key,
			ir_code_table[i].label,
			ir_code_table[i].descr);
	}
}

void do_ir_codes(char *keys) {
	int ix;
	char *p = keys;
	char buf[1024], *q;

	for (  ; *p; ++p) {
		*buf = '\0';
		ix = key2ircode(*p);
		if (ix >= 0) {
			q = ir_code_table[ix].code;
			strcat(buf, prefix);
			strcat(buf, q);
			strcat(buf, suffix);
			printf("%s (%s)\n", ir_code_table[ix].label, buf);
			if (debug) printf("%s\n", ircode2level(buf));
			send_ir(ircode2level(buf),
				(long)24, (long)in_loops, noints);
		} else {
			fprintf(stderr,
				"Can't find IR code for key '%c'!\n", *p);
		}
		send_ir("_", ((long)ir_sleep * 1000L) / 25L,
			(long)in_loops, 0);
	}
}

void usage(char *program)
{
	fprintf(stderr,
"usage: %s [options...] key_codes...\n"
"options:\n"
"   -d        - debug mode\n"
"   -c        - calibrate delay\n"
"   -h        - show all commands\n"
"   -p nnn    - use player \"nnn\" (default = 1)\n"
"   -ir l n   - IR test: l outer, n inner loops\n"
"   -l loops  - 40kHz loop count\n"
"   -r num    - repeat IR codes \"num\" times\n"
"   -keys     - show IR key codes\n"
"where \"command\" is of the form \"dddddddd\" (d=0/1)\n"
"example: ir 100111010001 (pause)\n"
	"", program);
	exit(1);
}

int main(int argc, char **argv)
{
	long delta_msecs;
	char *p, *q, *program;
	struct timeb t1, t2;  

	program = *argv;
	if ((p = strrchr(program, '\\')) != 0) program = p + 1;
	if ((p = strrchr(program, '.')) != 0) *p = '\0';
	--argc; ++argv;

	while (argc && **argv == '-') {
		if (!strcmp(*argv, "-")) {
			break;
		} else if (!strcmp(*argv, "-d")) {
			debug = 1;
			--argc; ++argv;
		} else if (!strcmp(*argv, "-h")) {
			usage(program);
			exit(0);
		} else if (!strcmp(*argv, "-keys")) {
			show_ir_table();
			exit(0);
		} else if (!strcmp(*argv, "-c")) {
			calibrate = 1;
			--argc; ++argv;
		} else if (!strcmp(*argv, "-l")) {
			--argc; ++argv;
			if (!argc) usage(program);
			in_loops = strtol(*argv, NULL, 0);
			if (debug)
				printf("40kHz loop count: %ld\n", in_loops);
			--argc; ++argv;
		} else if (!strcmp(*argv, "-r")) {
			--argc; ++argv;
			if (!argc) usage(program);
			repeat = strtol(*argv, NULL, 0);
			--argc; ++argv;
		} else if (!strcmp(*argv, "-ir")) {
			--argc; ++argv;
			if (!argc) usage(program);
			out_loops = strtol(*argv, NULL, 0);
			--argc; ++argv;
			if (!argc) usage(program);
			in_loops = strtol(*argv, NULL, 0);
			--argc; ++argv;
		} else if (!strcmp(*argv, "-p")) {
			--argc; ++argv;
			if (!argc) usage(program);
			player = atoi(*argv);
			if (debug)
				printf("player: %d\n", player);
			--argc; ++argv;
		} else {
			fprintf(stderr,
				"%s: unknown option (%s)!"
				"  (Use %s -h for assistance)\n",
				program, *argv, program);
			exit(1);

		}
	}

        if (!argc) {
		printf("IR test: %ld outer, %ld inner\n", out_loops, in_loops);
		ftime(&t1);  /* get start time */  
#if 0
		test_40khz(out_loops, in_loops);
		ftime(&t2);  /* get end time */  
		delta_msecs = (t2.time * 1000L + t2.millitm) -
		       	(t1.time * 1000L + t1.millitm);
		if (delta_msecs == 0) delta_msecs = 1;
		printf("%ld msec, %ld loops, %g msecs/out loop, %g KHz\n",
				delta_msecs, (long)out_loops * in_loops,
				((double)delta_msecs) / out_loops,
				((double)out_loops) / (double)delta_msecs);
		exit(0);
#else
		send_ir("__________",
			(long)(out_loops / 10L), (long)in_loops, 0);
		ftime(&t2);  /* get end time */  
		delta_msecs = (t2.time * 1000L + t2.millitm) -
		       	(t1.time * 1000L + t1.millitm);
		if (delta_msecs == 0) delta_msecs = 1;
		printf("%ld msec, %ld loops, %g msecs/out loop, %g KHz\n",
				delta_msecs, (long)out_loops * in_loops,
				((double)delta_msecs) / out_loops,
				((double)out_loops) / (double)delta_msecs);
		exit(0);
#endif
	} else {
		while (argc) {
			do_ir_codes(*argv);
			--argc; ++argv;
		}
	}
	return 0;
}
